import { AnalysisResult, DetectedPhrase } from '../types';
import { aiPhrases } from '../data/aiPhrases';

export const analyzeText = (
  text: string, 
  activeCategories: Record<string, boolean>
): AnalysisResult => {
  // Filter phrases based on active categories
  const filteredPhrases = aiPhrases.filter(phrase => 
    activeCategories[phrase.category]
  );
  
  const detectedPhrases: DetectedPhrase[] = [];
  const words = text.split(/\s+/).filter(word => word.length > 0);
  const wordCount = words.length;
  
  // Find all occurrences of AI phrases in the text
  for (const phrase of filteredPhrases) {
    const regex = new RegExp(phrase.phrase, 'gi');
    let match;
    
    while ((match = regex.exec(text)) !== null) {
      detectedPhrases.push({
        phrase: phrase.phrase,
        index: match.index,
        category: phrase.category,
      });
    }
  }
  
  // Calculate unique phrases
  const uniquePhrases = new Set(detectedPhrases.map(p => p.phrase));
  const uniquePhraseCount = uniquePhrases.size;
  
  // Calculate statistics
  const phraseCount = detectedPhrases.length;
  const phraseDensity = wordCount > 0 ? phraseCount / wordCount : 0;
  const uniquePhraseRatio = phraseCount > 0 ? uniquePhraseCount / phraseCount : 0;
  
  return {
    text,
    detectedPhrases,
    statistics: {
      wordCount,
      phraseCount,
      phraseDensity,
      uniquePhraseCount,
      uniquePhraseRatio,
    }
  };
};