import React from 'react';
import { Trash2 } from 'lucide-react';

interface TextInputProps {
  text: string;
  setText: (text: string) => void;
  isLoading: boolean;
}

export const TextInput: React.FC<TextInputProps> = ({ text, setText, isLoading }) => {
  const handleClear = () => {
    setText('');
  };

  const handlePaste = async () => {
    try {
      const clipboardText = await navigator.clipboard.readText();
      setText(clipboardText);
    } catch (err) {
      console.error('Failed to read clipboard: ', err);
    }
  };

  return (
    <div className="rounded-lg border bg-white dark:border-gray-800 dark:bg-gray-800">
      <div className="flex items-center justify-between border-b p-3 dark:border-gray-700">
        <h2 className="font-medium dark:text-white">Input Text</h2>
        <div className="flex gap-2">
          <button
            onClick={handlePaste}
            disabled={isLoading}
            className="rounded-md bg-blue-50 px-3 py-1 text-sm text-blue-700 transition-colors hover:bg-blue-100 disabled:cursor-not-allowed disabled:opacity-50 dark:bg-blue-900/20 dark:text-blue-400 dark:hover:bg-blue-900/30"
          >
            Paste from clipboard
          </button>
          <button
            onClick={handleClear}
            disabled={!text || isLoading}
            className="rounded-md px-2 text-gray-500 hover:bg-gray-100 disabled:cursor-not-allowed disabled:opacity-50 dark:text-gray-400 dark:hover:bg-gray-700"
          >
            <Trash2 size={18} />
          </button>
        </div>
      </div>
      <textarea
        value={text}
        onChange={(e) => setText(e.target.value)}
        disabled={isLoading}
        placeholder="Enter or paste text to analyze for AI-generated phrases..."
        className="min-h-[300px] w-full resize-y rounded-b-lg border-none bg-transparent p-4 focus:outline-none dark:text-white"
      ></textarea>
    </div>
  );
};