import React, { useState, useEffect } from 'react';
import { TextInput } from './TextInput';
import { HighlightedOutput } from './HighlightedOutput';
import { Statistics } from './Statistics';
import { analyzeText } from '../utils/textAnalysis';
import { AnalysisResult } from '../types';
import { Info, Upload, Download } from 'lucide-react';

export const TextAnalyzer: React.FC = () => {
  const [text, setText] = useState('');
  const [result, setResult] = useState<AnalysisResult | null>(null);
  const [isLoading, setIsLoading] = useState(false);
  const [activeCategories, setActiveCategories] = useState<Record<string, boolean>>({
    common: true,
    formal: true,
    technical: true,
    creative: true,
  });

  useEffect(() => {
    if (text.trim()) {
      setIsLoading(true);
      // Simulate processing delay
      const timer = setTimeout(() => {
        const analysisResult = analyzeText(text, activeCategories);
        setResult(analysisResult);
        setIsLoading(false);
      }, 300);
      
      return () => clearTimeout(timer);
    } else {
      setResult(null);
    }
  }, [text, activeCategories]);

  const handleExport = () => {
    if (!result) return;
    
    const exportData = {
      originalText: text,
      detectedPhrases: result.detectedPhrases,
      statistics: result.statistics,
      timestamp: new Date().toISOString(),
    };
    
    const blob = new Blob([JSON.stringify(exportData, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `ai-analysis-${new Date().toISOString().split('T')[0]}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  };

  const handleCategoryToggle = (category: string) => {
    setActiveCategories(prev => ({
      ...prev,
      [category]: !prev[category],
    }));
  };

  return (
    <div className="space-y-6">
      <div className="mb-6 rounded-lg bg-blue-50 p-4 dark:bg-blue-900/20">
        <div className="flex items-start gap-3">
          <div className="mt-0.5 text-blue-500 dark:text-blue-400">
            <Info size={20} />
          </div>
          <div>
            <h3 className="font-medium text-blue-600 dark:text-blue-400">About this tool</h3>
            <p className="mt-1 text-sm text-blue-700/70 dark:text-blue-300/70">
              Paste text to analyze and detect common phrases associated with AI-generated content. 
              This tool highlights potential AI phrases and provides statistics on the analysis.
            </p>
          </div>
        </div>
      </div>

      <div className="grid gap-6 lg:grid-cols-[1fr_1fr]">
        <div className="space-y-4">
          <TextInput text={text} setText={setText} isLoading={isLoading} />
          
          <div className="rounded-lg border bg-white p-4 dark:border-gray-800 dark:bg-gray-800">
            <h3 className="mb-3 font-medium">Phrase Categories</h3>
            <div className="flex flex-wrap gap-2">
              {Object.entries(activeCategories).map(([category, isActive]) => (
                <button
                  key={category}
                  onClick={() => handleCategoryToggle(category)}
                  className={`rounded-full px-3 py-1 text-sm transition-colors ${
                    isActive
                      ? 'bg-blue-100 text-blue-700 dark:bg-blue-900/30 dark:text-blue-400'
                      : 'bg-gray-100 text-gray-500 dark:bg-gray-700 dark:text-gray-400'
                  }`}
                >
                  {category.charAt(0).toUpperCase() + category.slice(1)}
                </button>
              ))}
            </div>
          </div>
        </div>

        <div className="space-y-4">
          {result && (
            <>
              <div className="flex justify-between">
                <h2 className="text-lg font-medium dark:text-white">Analysis Results</h2>
                <div className="space-x-2">
                  <button 
                    onClick={handleExport}
                    className="inline-flex items-center gap-1 rounded-md bg-blue-50 px-3 py-1 text-sm text-blue-700 transition-colors hover:bg-blue-100 dark:bg-blue-900/20 dark:text-blue-400 dark:hover:bg-blue-900/30"
                  >
                    <Download size={16} />
                    Export
                  </button>
                </div>
              </div>
              <Statistics result={result} />
              <HighlightedOutput result={result} />
            </>
          )}
          
          {!result && !isLoading && text.trim() === '' && (
            <div className="flex min-h-[300px] flex-col items-center justify-center rounded-lg border border-dashed bg-white dark:border-gray-700 dark:bg-gray-800">
              <Upload className="mb-2 text-gray-400" size={32} />
              <p className="text-gray-500 dark:text-gray-400">Paste some text to analyze</p>
            </div>
          )}
          
          {isLoading && (
            <div className="flex min-h-[300px] flex-col items-center justify-center rounded-lg border bg-white dark:border-gray-800 dark:bg-gray-800">
              <div className="h-6 w-6 animate-spin rounded-full border-b-2 border-t-2 border-blue-500"></div>
              <p className="mt-2 text-sm text-gray-500 dark:text-gray-400">Analyzing text...</p>
            </div>
          )}
        </div>
      </div>
    </div>
  );
};