import React from 'react';
import { AnalysisResult } from '../types';
import { BarChart, AlertTriangle } from 'lucide-react';

interface StatisticsProps {
  result: AnalysisResult;
}

export const Statistics: React.FC<StatisticsProps> = ({ result }) => {
  const { statistics } = result;
  
  // Calculate AI likelihood score (simplified version)
  const getAIScore = () => {
    const { phraseDensity, uniquePhraseRatio } = statistics;
    // Simple algorithm - can be made more sophisticated
    const score = (phraseDensity * 50) + (uniquePhraseRatio * 50);
    return Math.min(Math.max(score, 0), 100);
  };
  
  const aiScore = getAIScore();
  
  const getLikelihoodLabel = (score: number) => {
    if (score < 20) return 'Very Low';
    if (score < 40) return 'Low';
    if (score < 60) return 'Moderate';
    if (score < 80) return 'High';
    return 'Very High';
  };
  
  const getLikelihoodColor = (score: number) => {
    if (score < 30) return 'bg-green-500 dark:bg-green-600';
    if (score < 60) return 'bg-yellow-500 dark:bg-yellow-600';
    return 'bg-red-500 dark:bg-red-600';
  };

  return (
    <div className="rounded-lg border bg-white dark:border-gray-800 dark:bg-gray-800">
      <div className="border-b p-3 dark:border-gray-700">
        <h3 className="flex items-center gap-1 font-medium dark:text-white">
          <BarChart size={16} />
          Analysis Statistics
        </h3>
      </div>
      
      <div className="p-4">
        <div className="mb-4">
          <div className="mb-1 flex items-center justify-between">
            <h4 className="text-sm font-medium text-gray-700 dark:text-gray-300">AI Content Likelihood</h4>
            <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
              {getLikelihoodLabel(aiScore)}
            </span>
          </div>
          <div className="h-4 w-full overflow-hidden rounded-full bg-gray-200 dark:bg-gray-700">
            <div 
              className={`h-full transition-all ${getLikelihoodColor(aiScore)}`}
              style={{ width: `${aiScore}%` }}
            ></div>
          </div>
        </div>
        
        <div className="mb-4 grid grid-cols-2 gap-4">
          <div className="rounded-lg bg-gray-50 p-3 dark:bg-gray-700/50">
            <div className="text-xs text-gray-500 dark:text-gray-400">Total Words</div>
            <div className="text-xl font-bold dark:text-white">{statistics.wordCount}</div>
          </div>
          <div className="rounded-lg bg-gray-50 p-3 dark:bg-gray-700/50">
            <div className="text-xs text-gray-500 dark:text-gray-400">AI Phrases Detected</div>
            <div className="text-xl font-bold dark:text-white">{statistics.phraseCount}</div>
          </div>
        </div>

        <div className="space-y-2">
          <div className="grid grid-cols-2 gap-4">
            <div className="rounded-lg bg-gray-50 p-3 dark:bg-gray-700/50">
              <div className="text-xs text-gray-500 dark:text-gray-400">Phrase Density</div>
              <div className="text-lg font-bold dark:text-white">
                {(statistics.phraseDensity * 100).toFixed(1)}%
              </div>
            </div>
            <div className="rounded-lg bg-gray-50 p-3 dark:bg-gray-700/50">
              <div className="text-xs text-gray-500 dark:text-gray-400">Unique Phrases</div>
              <div className="text-lg font-bold dark:text-white">
                {statistics.uniquePhraseRatio < 1 
                  ? (statistics.uniquePhraseRatio * 100).toFixed(0) 
                  : statistics.uniquePhraseCount}
              </div>
            </div>
          </div>
        </div>
        
        {aiScore > 70 && (
          <div className="mt-4 rounded-lg bg-red-50 p-3 dark:bg-red-900/20">
            <div className="flex items-start gap-2">
              <AlertTriangle className="mt-0.5 text-red-500 dark:text-red-400" size={18} />
              <div>
                <p className="text-sm font-medium text-red-700 dark:text-red-400">
                  High likelihood of AI-generated content
                </p>
                <p className="mt-1 text-xs text-red-600/80 dark:text-red-300/80">
                  This text contains a high number of phrases commonly used by AI systems.
                </p>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};