import React, { useRef } from 'react';
import { AnalysisResult } from '../types';
import { Copy, Check } from 'lucide-react';

interface HighlightedOutputProps {
  result: AnalysisResult;
}

export const HighlightedOutput: React.FC<HighlightedOutputProps> = ({ result }) => {
  const [copied, setCopied] = React.useState(false);
  const outputRef = useRef<HTMLDivElement>(null);

  const handleCopyToClipboard = () => {
    if (outputRef.current) {
      const textToCopy = outputRef.current.innerText;
      navigator.clipboard.writeText(textToCopy).then(() => {
        setCopied(true);
        setTimeout(() => setCopied(false), 2000);
      });
    }
  };

  // Create highlighted text with spans around detected phrases
  const renderHighlightedText = () => {
    let lastIndex = 0;
    const segments = [];
    const sortedPhrases = [...result.detectedPhrases].sort((a, b) => a.index - b.index);

    for (const phrase of sortedPhrases) {
      // Add text before the phrase
      if (phrase.index > lastIndex) {
        segments.push(result.text.substring(lastIndex, phrase.index));
      }

      // Add the highlighted phrase
      segments.push(
        <span 
          key={`phrase-${phrase.index}`}
          className={`bg-yellow-200 px-0.5 py-0.5 dark:bg-yellow-900/60 ${
            phrase.category === 'common' ? 'bg-orange-200 dark:bg-orange-900/60' :
            phrase.category === 'formal' ? 'bg-blue-200 dark:bg-blue-900/60' :
            phrase.category === 'technical' ? 'bg-green-200 dark:bg-green-900/60' :
            'bg-purple-200 dark:bg-purple-900/60'
          }`}
          title={`Category: ${phrase.category}`}
        >
          {result.text.substring(phrase.index, phrase.index + phrase.phrase.length)}
        </span>
      );

      lastIndex = phrase.index + phrase.phrase.length;
    }

    // Add any remaining text
    if (lastIndex < result.text.length) {
      segments.push(result.text.substring(lastIndex));
    }

    return segments;
  };

  return (
    <div className="rounded-lg border bg-white dark:border-gray-800 dark:bg-gray-800">
      <div className="flex items-center justify-between border-b p-3 dark:border-gray-700">
        <h3 className="font-medium dark:text-white">Highlighted Result</h3>
        <button
          onClick={handleCopyToClipboard}
          className="inline-flex items-center gap-1 rounded-md bg-gray-100 px-3 py-1 text-sm text-gray-700 transition-colors hover:bg-gray-200 dark:bg-gray-700 dark:text-gray-300 dark:hover:bg-gray-600"
        >
          {copied ? <Check size={16} /> : <Copy size={16} />}
          {copied ? 'Copied!' : 'Copy'}
        </button>
      </div>
      
      <div 
        ref={outputRef}
        className="min-h-[200px] max-h-[400px] overflow-auto p-4 text-gray-700 dark:text-gray-200"
      >
        {result.detectedPhrases.length > 0 ? (
          <div>{renderHighlightedText()}</div>
        ) : (
          <p className="italic text-gray-500 dark:text-gray-400">
            No AI-associated phrases detected in this text.
          </p>
        )}
      </div>
      
      <div className="border-t p-3 text-xs text-gray-500 dark:border-gray-700 dark:text-gray-400">
        <div className="flex flex-wrap gap-2">
          <span className="inline-flex items-center gap-1">
            <span className="inline-block h-3 w-3 rounded-sm bg-orange-200 dark:bg-orange-900/60"></span>
            Common
          </span>
          <span className="inline-flex items-center gap-1">
            <span className="inline-block h-3 w-3 rounded-sm bg-blue-200 dark:bg-blue-900/60"></span>
            Formal
          </span>
          <span className="inline-flex items-center gap-1">
            <span className="inline-block h-3 w-3 rounded-sm bg-green-200 dark:bg-green-900/60"></span>
            Technical
          </span>
          <span className="inline-flex items-center gap-1">
            <span className="inline-block h-3 w-3 rounded-sm bg-purple-200 dark:bg-purple-900/60"></span>
            Creative
          </span>
        </div>
      </div>
    </div>
  );
};