document.addEventListener('DOMContentLoaded', () => {
  // Dark mode toggle
  const darkModeToggle = document.getElementById('darkModeToggle');
  const isDarkMode = localStorage.getItem('darkMode') === 'true' || 
    window.matchMedia('(prefers-color-scheme: dark)').matches;
  
  if (isDarkMode) {
    document.documentElement.classList.add('dark');
  }

  darkModeToggle.addEventListener('click', () => {
    document.documentElement.classList.toggle('dark');
    localStorage.setItem('darkMode', document.documentElement.classList.contains('dark'));
  });

  // Set current year in footer
  document.getElementById('currentYear').textContent = new Date().getFullYear();

  // Input elements
  const inputText = document.getElementById('inputText');
  const pasteButton = document.getElementById('pasteButton');
  const clearButton = document.getElementById('clearButton');
  const resultsContainer = document.getElementById('resultsContainer');
  
  // Category toggles
  const categoryToggles = document.querySelectorAll('.category-toggle');
  const activeCategories = {
    common: true,
    formal: true,
    technical: true,
    creative: true
  };

  // Analysis functions
  function analyzeText(text, activeCategories) {
    const filteredPhrases = aiPhrases.filter(phrase => activeCategories[phrase.category]);
    const detectedPhrases = [];
    const words = text.split(/\s+/).filter(word => word.length > 0);
    const wordCount = words.length;
    
    for (const phrase of filteredPhrases) {
      const regex = new RegExp(phrase.phrase, 'gi');
      let match;
      
      while ((match = regex.exec(text)) !== null) {
        detectedPhrases.push({
          phrase: phrase.phrase,
          index: match.index,
          category: phrase.category
        });
      }
    }
    
    const uniquePhrases = new Set(detectedPhrases.map(p => p.phrase));
    const uniquePhraseCount = uniquePhrases.size;
    const phraseCount = detectedPhrases.length;
    const phraseDensity = wordCount > 0 ? phraseCount / wordCount : 0;
    const uniquePhraseRatio = phraseCount > 0 ? uniquePhraseCount / phraseCount : 0;
    
    return {
      text,
      detectedPhrases,
      statistics: {
        wordCount,
        phraseCount,
        phraseDensity,
        uniquePhraseCount,
        uniquePhraseRatio
      }
    };
  }

  function getAIScore(statistics) {
    const { phraseDensity, uniquePhraseRatio } = statistics;
    const score = (phraseDensity * 50) + (uniquePhraseRatio * 50);
    return Math.min(Math.max(score, 0), 100);
  }

  function getLikelihoodLabel(score) {
    if (score < 20) return 'Very Low';
    if (score < 40) return 'Low';
    if (score < 60) return 'Moderate';
    if (score < 80) return 'High';
    return 'Very High';
  }

  function getLikelihoodColor(score) {
    if (score < 30) return 'bg-green-500 dark:bg-green-600';
    if (score < 60) return 'bg-yellow-500 dark:bg-yellow-600';
    return 'bg-red-500 dark:bg-red-600';
  }

  function renderResults(result) {
    const aiScore = getAIScore(result.statistics);
    
    let html = `
      <div class="flex justify-between">
        <h2 class="text-lg font-medium dark:text-white">Analysis Results</h2>
        <div class="space-x-2">
          <button onclick="handleExport()" class="inline-flex items-center gap-1 rounded-md bg-blue-50 px-3 py-1 text-sm text-blue-700 transition-colors hover:bg-blue-100 dark:bg-blue-900/20 dark:text-blue-400 dark:hover:bg-blue-900/30">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
              <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
              <polyline points="7 10 12 15 17 10"></polyline>
              <line x1="12" y1="15" x2="12" y2="3"></line>
            </svg>
            Export
          </button>
        </div>
      </div>

      <div class="rounded-lg border bg-white dark:border-gray-800 dark:bg-gray-800">
        <div class="border-b p-3 dark:border-gray-700">
          <h3 class="flex items-center gap-1 font-medium dark:text-white">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
              <path d="M3 3v18h18"></path>
              <path d="M18 17V9"></path>
              <path d="M13 17V5"></path>
              <path d="M8 17v-3"></path>
            </svg>
            Analysis Statistics
          </h3>
        </div>
        
        <div class="p-4">
          <div class="mb-4">
            <div class="mb-1 flex items-center justify-between">
              <h4 class="text-sm font-medium text-gray-700 dark:text-gray-300">AI Content Likelihood</h4>
              <span class="text-sm font-medium text-gray-700 dark:text-gray-300">
                ${getLikelihoodLabel(aiScore)}
              </span>
            </div>
            <div class="h-4 w-full overflow-hidden rounded-full bg-gray-200 dark:bg-gray-700">
              <div class="h-full transition-all ${getLikelihoodColor(aiScore)}" style="width: ${aiScore}%"></div>
            </div>
          </div>
          
          <div class="mb-4 grid grid-cols-2 gap-4">
            <div class="rounded-lg bg-gray-50 p-3 dark:bg-gray-700/50">
              <div class="text-xs text-gray-500 dark:text-gray-400">Total Words</div>
              <div class="text-xl font-bold dark:text-white">${result.statistics.wordCount}</div>
            </div>
            <div class="rounded-lg bg-gray-50 p-3 dark:bg-gray-700/50">
              <div class="text-xs text-gray-500 dark:text-gray-400">AI Phrases Detected</div>
              <div class="text-xl font-bold dark:text-white">${result.statistics.phraseCount}</div>
            </div>
          </div>

          <div class="space-y-2">
            <div class="grid grid-cols-2 gap-4">
              <div class="rounded-lg bg-gray-50 p-3 dark:bg-gray-700/50">
                <div class="text-xs text-gray-500 dark:text-gray-400">Phrase Density</div>
                <div class="text-lg font-bold dark:text-white">
                  ${(result.statistics.phraseDensity * 100).toFixed(1)}%
                </div>
              </div>
              <div class="rounded-lg bg-gray-50 p-3 dark:bg-gray-700/50">
                <div class="text-xs text-gray-500 dark:text-gray-400">Unique Phrases</div>
                <div class="text-lg font-bold dark:text-white">
                  ${result.statistics.uniquePhraseRatio < 1 
                    ? (result.statistics.uniquePhraseRatio * 100).toFixed(0) 
                    : result.statistics.uniquePhraseCount}
                </div>
              </div>
            </div>
          </div>
          
          ${aiScore > 70 ? `
            <div class="mt-4 rounded-lg bg-red-50 p-3 dark:bg-red-900/20">
              <div class="flex items-start gap-2">
                <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="mt-0.5 text-red-500 dark:text-red-400">
                  <path d="m21.73 18-8-14a2 2 0 0 0-3.48 0l-8 14A2 2 0 0 0 4 21h16a2 2 0 0 0 1.73-3Z"></path>
                  <path d="M12 9v4"></path>
                  <path d="M12 17h.01"></path>
                </svg>
                <div>
                  <p class="text-sm font-medium text-red-700 dark:text-red-400">
                    High likelihood of AI-generated content
                  </p>
                  <p class="mt-1 text-xs text-red-600/80 dark:text-red-300/80">
                    This text contains a high number of phrases commonly used by AI systems.
                  </p>
                </div>
              </div>
            </div>
          ` : ''}
        </div>
      </div>

      <div class="rounded-lg border bg-white dark:border-gray-800 dark:bg-gray-800">
        <div class="flex items-center justify-between border-b p-3 dark:border-gray-700">
          <h3 class="font-medium dark:text-white">Highlighted Result</h3>
          <button onclick="handleCopyToClipboard()" class="inline-flex items-center gap-1 rounded-md bg-gray-100 px-3 py-1 text-sm text-gray-700 transition-colors hover:bg-gray-200 dark:bg-gray-700 dark:text-gray-300 dark:hover:bg-gray-600">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
              <rect width="14" height="14" x="8" y="8" rx="2" ry="2"></rect>
              <path d="M4 16c-1.1 0-2-.9-2-2V4c0-1.1.9-2 2-2h10c1.1 0 2 .9 2 2"></path>
            </svg>
            Copy
          </button>
        </div>
        
        <div id="highlightedText" class="min-h-[200px] max-h-[400px] overflow-auto p-4 text-gray-700 dark:text-gray-200">
          ${renderHighlightedText(result)}
        </div>
        
        <div class="border-t p-3 text-xs text-gray-500 dark:border-gray-700 dark:text-gray-400">
          <div class="flex flex-wrap gap-2">
            <span class="inline-flex items-center gap-1">
              <span class="inline-block h-3 w-3 rounded-sm bg-orange-200 dark:bg-orange-900/60"></span>
              Common
            </span>
            <span class="inline-flex items-center gap-1">
              <span class="inline-block h-3 w-3 rounded-sm bg-blue-200 dark:bg-blue-900/60"></span>
              Formal
            </span>
            <span class="inline-flex items-center gap-1">
              <span class="inline-block h-3 w-3 rounded-sm bg-green-200 dark:bg-green-900/60"></span>
              Technical
            </span>
            <span class="inline-flex items-center gap-1">
              <span class="inline-block h-3 w-3 rounded-sm bg-purple-200 dark:bg-purple-900/60"></span>
              Creative
            </span>
          </div>
        </div>
      </div>
    `;
    
    resultsContainer.innerHTML = html;
  }

  function renderHighlightedText(result) {
    if (result.detectedPhrases.length === 0) {
      return `<p class="italic text-gray-500 dark:text-gray-400">
        No AI-associated phrases detected in this text.
      </p>`;
    }

    let lastIndex = 0;
    let html = '';
    const sortedPhrases = [...result.detectedPhrases].sort((a, b) => a.index - b.index);

    for (const phrase of sortedPhrases) {
      if (phrase.index > lastIndex) {
        html += result.text.substring(lastIndex, phrase.index);
      }

      const highlightClass = 
        phrase.category === 'common' ? 'bg-orange-200 dark:bg-orange-900/60' :
        phrase.category === 'formal' ? 'bg-blue-200 dark:bg-blue-900/60' :
        phrase.category === 'technical' ? 'bg-green-200 dark:bg-green-900/60' :
        'bg-purple-200 dark:bg-purple-900/60';

      html += `<span class="${highlightClass} px-0.5 py-0.5" title="Category: ${phrase.category}">
        ${result.text.substring(phrase.index, phrase.index + phrase.phrase.length)}
      </span>`;

      lastIndex = phrase.index + phrase.phrase.length;
    }

    if (lastIndex < result.text.length) {
      html += result.text.substring(lastIndex);
    }

    return html;
  }

  // Event handlers
  function handleInput() {
    const text = inputText.value.trim();
    if (text) {
      const result = analyzeText(text, activeCategories);
      renderResults(result);
    } else {
      resultsContainer.innerHTML = `
        <div class="flex min-h-[300px] flex-col items-center justify-center rounded-lg border border-dashed bg-white dark:border-gray-700 dark:bg-gray-800">
          <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="mb-2 text-gray-400">
            <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
            <polyline points="17 8 12 3 7 8"></polyline>
            <line x1="12" y1="3" x2="12" y2="15"></line>
          </svg>
          <p class="text-gray-500 dark:text-gray-400">Paste some text to analyze</p>
        </div>
      `;
    }
  }

  window.handleCopyToClipboard = () => {
    const highlightedText = document.getElementById('highlightedText');
    if (highlightedText) {
      const textToCopy = highlightedText.innerText;
      navigator.clipboard.writeText(textToCopy).then(() => {
        const copyButton = highlightedText.previousElementSibling.querySelector('button');
        const originalContent = copyButton.innerHTML;
        copyButton.innerHTML = `
          <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <polyline points="20 6 9 17 4 12"></polyline>
          </svg>
          Copied!
        `;
        setTimeout(() => {
          copyButton.innerHTML = originalContent;
        }, 2000);
      });
    }
  };

  window.handleExport = () => {
    const text = inputText.value.trim();
    if (!text) return;
    
    const result = analyzeText(text, activeCategories);
    const exportData = {
      originalText: text,
      detectedPhrases: result.detectedPhrases,
      statistics: result.statistics,
      timestamp: new Date().toISOString()
    };
    
    const blob = new Blob([JSON.stringify(exportData, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `ai-analysis-${new Date().toISOString().split('T')[0]}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  };

  // Event listeners
  inputText.addEventListener('input', handleInput);
  
  pasteButton.addEventListener('click', async () => {
    try {
      const clipboardText = await navigator.clipboard.readText();
      inputText.value = clipboardText;
      handleInput();
    } catch (err) {
      console.error('Failed to read clipboard: ', err);
    }
  });
  
  clearButton.addEventListener('click', () => {
    inputText.value = '';
    handleInput();
  });
  
  categoryToggles.forEach(toggle => {
    toggle.addEventListener('click', () => {
      const category = toggle.dataset.category;
      activeCategories[category] = !activeCategories[category];
      toggle.classList.toggle('bg-blue-100');
      toggle.classList.toggle('text-blue-700');
      toggle.classList.toggle('dark:bg-blue-900/30');
      toggle.classList.toggle('dark:text-blue-400');
      toggle.classList.toggle('bg-gray-100');
      toggle.classList.toggle('text-gray-500');
      toggle.classList.toggle('dark:bg-gray-700');
      toggle.classList.toggle('dark:text-gray-400');
      handleInput();
    });
  });
});