import React, { useState, useEffect } from 'react';
import { Copy, RotateCcw, Wand2, ArrowRight, FileText, FileCheck, Brain, Shield, UserCheck, Sparkles } from 'lucide-react';
import TextInput from './TextInput';
import TextOutput from './TextOutput';
import SanitizerSettings from './SanitizerSettings';
import { sanitizeText } from '../utils/sanitizer';
import { SanitizerOptions } from '../types';

const TextSanitizer: React.FC = () => {
  const [inputText, setInputText] = useState('');
  const [outputText, setOutputText] = useState('');
  const [showSettings, setShowSettings] = useState(true);
  const [copySuccess, setCopySuccess] = useState(false);
  const [isProcessing, setIsProcessing] = useState(false);
  const [options, setOptions] = useState<SanitizerOptions>({
    removeExtraSpaces: true,
    normalizeNewlines: true,
    removeHtmlTags: true,
    convertUrlsToLinks: false,
    capitalizeFirstLetter: false,
    removeEmptyLines: true,
    removeBlankLines: true,
    removeLeadingSpaces: true,
    removeTrailingSpaces: true,
    removePunctuation: false,
    stripEmojis: false
  });

  useEffect(() => {
    const sanitized = sanitizeText(inputText, options);
    setOutputText(sanitized);
  }, [inputText, options]);

  const handleCopyToClipboard = () => {
    navigator.clipboard.writeText(outputText);
    setCopySuccess(true);
    setTimeout(() => setCopySuccess(false), 2000);
  };

  const handleReset = () => {
    setInputText('');
    setOutputText('');
  };

  const getStats = (text: string) => {
    const chars = text.length;
    const words = text.trim() ? text.trim().split(/\s+/).length : 0;
    const lines = text.trim() ? text.split(/\r\n|\r|\n/).length : 0;
    return { chars, words, lines };
  };

  const handleAIAction = (action: string) => {
    setIsProcessing(true);
    setTimeout(() => {
      let processedText = inputText;
      switch (action) {
        case 'humanize':
          processedText = "This would integrate with an AI service to make the text more human-like";
          break;
        case 'plagiarism':
          processedText = "This would check for plagiarism using an AI service";
          break;
        case 'improve':
          processedText = "This would use AI to improve the writing quality";
          break;
        case 'summarize':
          processedText = "This would generate an AI-powered summary of the text";
          break;
      }
      setOutputText(processedText);
      setIsProcessing(false);
    }, 1500);
  };

  const inputStats = getStats(inputText);
  const outputStats = getStats(outputText);

  return (
    <div className="max-w-6xl mx-auto">
      <div className="mb-8 text-center">
        <div className="inline-flex items-center justify-center space-x-3 mb-4">
          <FileText className="h-8 w-8 text-blue-500" />
          <ArrowRight className="h-6 w-6 text-gray-400" />
          <Wand2 className="h-8 w-8 text-blue-500 animate-bounce" />
          <ArrowRight className="h-6 w-6 text-gray-400" />
          <FileCheck className="h-8 w-8 text-green-500" />
        </div>
        <h2 className="text-3xl font-bold text-gray-800 mb-3">AI Text Sanitizer</h2>
        <p className="text-gray-600 max-w-2xl mx-auto">
          Transform messy text into clean, well-formatted content. Our AI-powered sanitizer removes unwanted formatting
          while preserving your text's essential layout and structure.
        </p>
      </div>

      {/* Top Banner Ad */}
      <div className="w-full h-32 bg-gradient-to-r from-gray-50 to-blue-50 rounded-xl mb-8 shadow-lg border border-blue-100 overflow-hidden relative group hover:shadow-xl transition-all duration-300">
        <div className="absolute inset-0 bg-blue-500/5 group-hover:bg-blue-500/10 transition-all duration-300"></div>
        <div className="absolute top-2 left-3 text-xs text-gray-400">Advertisement</div>
        <div className="w-full h-full flex items-center justify-center">
          <div className="text-center">
            <p className="text-blue-600 font-medium">Premium Banner Ad Space</p>
            <p className="text-sm text-gray-500 mt-1">728x90 Banner</p>
          </div>
        </div>
      </div>

      <div className="mb-8">
        <SanitizerSettings options={options} onChange={setOptions} />
      </div>

      <div className="flex flex-col lg:flex-row gap-8">
        <div className="flex-1">
          <div className="flex items-center justify-between mb-3">
            <div className="flex items-center space-x-2">
              <FileText className="h-5 w-5 text-blue-500" />
              <h3 className="text-lg font-medium text-gray-700">Input Text</h3>
            </div>
            <div className="text-sm text-gray-500 bg-gray-100 px-3 py-1 rounded-full">
              {inputStats.chars} chars | {inputStats.words} words | {inputStats.lines} lines
            </div>
          </div>
          <TextInput value={inputText} onChange={setInputText} />
        </div>

        <div className="flex-1">
          <div className="flex items-center justify-between mb-3">
            <div className="flex items-center space-x-2">
              <FileCheck className="h-5 w-5 text-green-500" />
              <h3 className="text-lg font-medium text-gray-700">Sanitized Output</h3>
            </div>
            <div className="text-sm text-gray-500 bg-gray-100 px-3 py-1 rounded-full">
              {outputStats.chars} chars | {outputStats.words} words | {outputStats.lines} lines
            </div>
          </div>
          <TextOutput value={outputText} />
          
          <div className="mt-4 flex justify-end space-x-2">
            <button
              onClick={handleReset}
              disabled={!inputText}
              className={`px-4 py-2 rounded-full text-sm flex items-center gap-2 ${
                !inputText
                  ? 'bg-gray-100 text-gray-400 cursor-not-allowed'
                  : 'bg-white border border-gray-300 text-gray-700 hover:bg-gray-50 hover:border-red-300 transition-all duration-200'
              }`}
            >
              <RotateCcw className="h-4 w-4" />
              <span>Reset</span>
            </button>
            <button
              onClick={handleCopyToClipboard}
              disabled={!outputText}
              className={`px-4 py-2 rounded-full text-white text-sm flex items-center gap-2 transition-all duration-200 ${
                !outputText
                  ? 'bg-blue-300 cursor-not-allowed'
                  : copySuccess
                  ? 'bg-green-500'
                  : 'bg-blue-500 hover:bg-blue-600'
              }`}
            >
              <Copy className="h-4 w-4" />
              <span>{copySuccess ? 'Copied!' : 'Copy to Clipboard'}</span>
            </button>
          </div>
        </div>
      </div>

      <div className="mt-8 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
        <button
          onClick={() => handleAIAction('plagiarism')}
          disabled={!inputText || isProcessing}
          className="flex items-center justify-center gap-2 px-4 py-3 bg-purple-500 hover:bg-purple-600 text-white rounded-xl transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed"
        >
          <Shield className="h-5 w-5" />
          <span>Check Plagiarism</span>
        </button>
        <button
          onClick={() => handleAIAction('humanize')}
          disabled={!inputText || isProcessing}
          className="flex items-center justify-center gap-2 px-4 py-3 bg-green-500 hover:bg-green-600 text-white rounded-xl transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed"
        >
          <UserCheck className="h-5 w-5" />
          <span>Make Human-like</span>
        </button>
        <button
          onClick={() => handleAIAction('improve')}
          disabled={!inputText || isProcessing}
          className="flex items-center justify-center gap-2 px-4 py-3 bg-blue-500 hover:bg-blue-600 text-white rounded-xl transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed"
        >
          <Sparkles className="h-5 w-5" />
          <span>Improve Writing</span>
        </button>
        <button
          onClick={() => handleAIAction('summarize')}
          disabled={!inputText || isProcessing}
          className="flex items-center justify-center gap-2 px-4 py-3 bg-orange-500 hover:bg-orange-600 text-white rounded-xl transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed"
        >
          <Brain className="h-5 w-5" />
          <span>Summarize Text</span>
        </button>
      </div>

      <div className="mt-4 text-center text-sm text-gray-500 bg-gray-50 rounded-lg p-4">
        <p>Some features may include affiliate links. We may earn a commission for purchases made through these links. 
        This helps support our service and enables us to keep providing high-quality text processing tools.</p>
      </div>

      {/* Bottom Banner Ad */}
      <div className="w-full h-32 bg-gradient-to-r from-gray-50 to-blue-50 rounded-xl mt-8 shadow-lg border border-blue-100 overflow-hidden relative group hover:shadow-xl transition-all duration-300">
        <div className="absolute inset-0 bg-blue-500/5 group-hover:bg-blue-500/10 transition-all duration-300"></div>
        <div className="absolute top-2 left-3 text-xs text-gray-400">Advertisement</div>
        <div className="w-full h-full flex items-center justify-center">
          <div className="text-center">
            <p className="text-blue-600 font-medium">Premium Banner Ad Space</p>
            <p className="text-sm text-gray-500 mt-1">728x90 Banner</p>
          </div>
        </div>
      </div>
    </div>
  );
};

export default TextSanitizer;