import React from 'react';
import { SanitizerOptions } from '../types';
import { Eraser, Link, Type, Sparkles, Hash, Smile } from 'lucide-react';

interface SanitizerSettingsProps {
  options: SanitizerOptions;
  onChange: (options: SanitizerOptions) => void;
}

const SanitizerSettings: React.FC<SanitizerSettingsProps> = ({ options, onChange }) => {
  const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, checked } = e.target;
    onChange({
      ...options,
      [name]: checked,
    });
  };

  return (
    <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
      <h3 className="text-lg font-medium text-gray-700 mb-4 flex items-center gap-2">
        <Sparkles className="h-5 w-5 text-blue-500" />
        Text Cleaning Options
      </h3>
      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
        <OptionToggle
          id="removeExtraSpaces"
          icon={<Eraser className="h-4 w-4 text-blue-500" />}
          label="Remove Extra Spaces"
          description="Clean up multiple spaces"
          checked={options.removeExtraSpaces}
          onChange={handleChange}
        />
        <OptionToggle
          id="removeLeadingSpaces"
          icon={<Eraser className="h-4 w-4 text-blue-500" />}
          label="Remove Leading Spaces"
          description="Clean start of lines"
          checked={options.removeLeadingSpaces}
          onChange={handleChange}
        />
        <OptionToggle
          id="removeTrailingSpaces"
          icon={<Eraser className="h-4 w-4 text-blue-500" />}
          label="Remove Trailing Spaces"
          description="Clean end of lines"
          checked={options.removeTrailingSpaces}
          onChange={handleChange}
        />
        <OptionToggle
          id="normalizeNewlines"
          icon={<Type className="h-4 w-4 text-blue-500" />}
          label="Fix Line Breaks"
          description="Standardize line endings"
          checked={options.normalizeNewlines}
          onChange={handleChange}
        />
        <OptionToggle
          id="removeHtmlTags"
          icon={<Eraser className="h-4 w-4 text-blue-500" />}
          label="Remove HTML"
          description="Strip all HTML tags"
          checked={options.removeHtmlTags}
          onChange={handleChange}
        />
        <OptionToggle
          id="convertUrlsToLinks"
          icon={<Link className="h-4 w-4 text-blue-500" />}
          label="Convert URLs to Links"
          description="Make URLs clickable"
          checked={options.convertUrlsToLinks}
          onChange={handleChange}
        />
        <OptionToggle
          id="capitalizeFirstLetter"
          icon={<Type className="h-4 w-4 text-blue-500" />}
          label="Auto-Capitalize"
          description="Capitalize sentences"
          checked={options.capitalizeFirstLetter}
          onChange={handleChange}
        />
        <OptionToggle
          id="removeEmptyLines"
          icon={<Eraser className="h-4 w-4 text-blue-500" />}
          label="Remove Empty Lines"
          description="Remove blank lines"
          checked={options.removeEmptyLines}
          onChange={handleChange}
        />
        <OptionToggle
          id="removePunctuation"
          icon={<Hash className="h-4 w-4 text-blue-500" />}
          label="Remove Punctuation"
          description="Strip punctuation marks"
          checked={options.removePunctuation}
          onChange={handleChange}
        />
        <OptionToggle
          id="stripEmojis"
          icon={<Smile className="h-4 w-4 text-blue-500" />}
          label="Strip Emojis"
          description="Remove emoji characters"
          checked={options.stripEmojis}
          onChange={handleChange}
        />
      </div>
    </div>
  );
};

interface OptionToggleProps {
  id: string;
  icon: React.ReactNode;
  label: string;
  description: string;
  checked: boolean;
  onChange: (e: React.ChangeEvent<HTMLInputElement>) => void;
}

const OptionToggle: React.FC<OptionToggleProps> = ({
  id,
  icon,
  label,
  description,
  checked,
  onChange
}) => {
  return (
    <div className="flex items-start space-x-3 p-3 rounded-lg bg-gray-50 hover:bg-gray-100 transition-colors duration-200">
      <div className="flex items-center h-5">
        <input
          id={id}
          name={id}
          type="checkbox"
          checked={checked}
          onChange={onChange}
          className="h-4 w-4 text-blue-500 border-gray-300 rounded focus:ring-blue-500"
        />
      </div>
      <div className="flex-1">
        <label htmlFor={id} className="flex items-center gap-2 font-medium text-gray-700 mb-1">
          {icon}
          {label}
        </label>
        <p className="text-sm text-gray-500">{description}</p>
      </div>
    </div>
  );
};

export default SanitizerSettings;