// Sanitizer options
const defaultOptions = {
    removeExtraSpaces: true,
    normalizeNewlines: true,
    removeHtmlTags: true,
    convertUrlsToLinks: false,
    capitalizeFirstLetter: false,
    removeEmptyLines: true,
    removeBlankLines: true,
    removeLeadingSpaces: true,
    removeTrailingSpaces: true,
    removePunctuation: false,
    stripEmojis: false
};

// Option definitions for the UI
const optionDefinitions = [
    {
        id: 'removeExtraSpaces',
        icon: 'eraser',
        label: 'Remove Extra Spaces',
        description: 'Clean up multiple spaces'
    },
    {
        id: 'removeLeadingSpaces',
        icon: 'eraser',
        label: 'Remove Leading Spaces',
        description: 'Clean start of lines'
    },
    {
        id: 'removeTrailingSpaces',
        icon: 'eraser',
        label: 'Remove Trailing Spaces',
        description: 'Clean end of lines'
    },
    {
        id: 'normalizeNewlines',
        icon: 'type',
        label: 'Fix Line Breaks',
        description: 'Standardize line endings'
    },
    {
        id: 'removeHtmlTags',
        icon: 'eraser',
        label: 'Remove HTML',
        description: 'Strip all HTML tags'
    },
    {
        id: 'convertUrlsToLinks',
        icon: 'link',
        label: 'Convert URLs to Links',
        description: 'Make URLs clickable'
    },
    {
        id: 'capitalizeFirstLetter',
        icon: 'type',
        label: 'Auto-Capitalize',
        description: 'Capitalize sentences'
    },
    {
        id: 'removeEmptyLines',
        icon: 'eraser',
        label: 'Remove Empty Lines',
        description: 'Remove blank lines'
    },
    {
        id: 'removePunctuation',
        icon: 'hash',
        label: 'Remove Punctuation',
        description: 'Strip punctuation marks'
    },
    {
        id: 'stripEmojis',
        icon: 'smile',
        label: 'Strip Emojis',
        description: 'Remove emoji characters'
    }
];

// DOM Elements
const inputText = document.getElementById('input-text');
const outputText = document.getElementById('output-text');
const optionsContainer = document.getElementById('options');
const resetBtn = document.getElementById('reset-btn');
const copyBtn = document.getElementById('copy-btn');
const inputStats = document.getElementById('input-stats');
const outputStats = document.getElementById('output-stats');

let currentOptions = { ...defaultOptions };

// Initialize options UI
function initializeOptions() {
    optionDefinitions.forEach(option => {
        const optionElement = document.createElement('div');
        optionElement.className = 'flex items-start space-x-3 p-3 rounded-lg bg-gray-50 hover:bg-gray-100 transition-colors duration-200';
        optionElement.innerHTML = `
            <div class="flex items-center h-5">
                <input
                    id="${option.id}"
                    type="checkbox"
                    ${currentOptions[option.id] ? 'checked' : ''}
                    class="h-4 w-4 text-blue-500 border-gray-300 rounded focus:ring-blue-500"
                >
            </div>
            <div class="flex-1">
                <label for="${option.id}" class="flex items-center gap-2 font-medium text-gray-700 mb-1">
                    <i class="lucide-${option.icon} h-4 w-4 text-blue-500"></i>
                    ${option.label}
                </label>
                <p class="text-sm text-gray-500">${option.description}</p>
            </div>
        `;
        optionsContainer.appendChild(optionElement);

        const checkbox = optionElement.querySelector(`#${option.id}`);
        checkbox.addEventListener('change', (e) => {
            currentOptions[option.id] = e.target.checked;
            processText();
        });
    });
}

// Text processing functions
function sanitizeText(text, options) {
    if (!text) return '';
    
    let result = text;

    if (options.removeHtmlTags) {
        result = result.replace(/<[^>]*>/g, '');
    }

    if (options.convertUrlsToLinks) {
        result = result.replace(
            /(https?:\/\/[^\s]+)/g,
            '<a href="$1" target="_blank" rel="noopener noreferrer">$1</a>'
        );
    }

    if (options.stripEmojis) {
        result = result.replace(/[\u{1F300}-\u{1F6FF}\u{1F900}-\u{1F9FF}]/gu, '');
    }

    if (options.removePunctuation) {
        result = result.replace(/[.,\/#!$%\^&\*;:{}=\-_`~()]/g, '');
    }

    if (options.normalizeNewlines) {
        result = result.replace(/\r\n|\r/g, '\n');
    }

    if (options.removeEmptyLines) {
        result = result.split('\n').filter(line => line.trim() !== '').join('\n');
    }

    if (options.removeLeadingSpaces) {
        result = result.split('\n').map(line => line.trimStart()).join('\n');
    }

    if (options.removeTrailingSpaces) {
        result = result.split('\n').map(line => line.trimEnd()).join('\n');
    }

    if (options.removeExtraSpaces) {
        result = result.replace(/\s+/g, ' ').trim();
    }

    if (options.capitalizeFirstLetter) {
        result = result.replace(
            /(^\w|\.\s+\w)/g,
            letter => letter.toUpperCase()
        );
    }

    return result;
}

function getStats(text) {
    const chars = text.length;
    const words = text.trim() ? text.trim().split(/\s+/).length : 0;
    const lines = text.trim() ? text.split(/\r\n|\r|\n/).length : 0;
    return `${chars} chars | ${words} words | ${lines} lines`;
}

function processText() {
    const processed = sanitizeText(inputText.value, currentOptions);
    outputText.value = processed;
    updateStats();
}

function updateStats() {
    inputStats.textContent = getStats(inputText.value);
    outputStats.textContent = getStats(outputText.value);
}

// Event Listeners
inputText.addEventListener('input', processText);

resetBtn.addEventListener('click', () => {
    inputText.value = '';
    outputText.value = '';
    updateStats();
});

copyBtn.addEventListener('click', async () => {
    if (!outputText.value) return;
    
    try {
        await navigator.clipboard.writeText(outputText.value);
        const originalText = copyBtn.innerHTML;
        copyBtn.innerHTML = '<i class="lucide-check h-4 w-4"></i><span>Copied!</span>';
        copyBtn.classList.remove('bg-blue-500', 'hover:bg-blue-600');
        copyBtn.classList.add('bg-green-500', 'hover:bg-green-600');
        
        setTimeout(() => {
            copyBtn.innerHTML = originalText;
            copyBtn.classList.remove('bg-green-500', 'hover:bg-green-600');
            copyBtn.classList.add('bg-blue-500', 'hover:bg-blue-600');
        }, 2000);
    } catch (err) {
        console.error('Failed to copy text:', err);
    }
});

// AI action buttons
document.querySelectorAll('.ai-action').forEach(button => {
    button.addEventListener('click', () => {
        if (!inputText.value) return;
        
        const action = button.querySelector('span').textContent;
        const originalText = button.innerHTML;
        button.innerHTML = '<i class="lucide-loader-2 h-5 w-5 animate-spin"></i><span>Processing...</span>';
        button.disabled = true;
        
        // Simulate AI processing
        setTimeout(() => {
            outputText.value = `This would integrate with an AI service to ${action.toLowerCase()}`;
            button.innerHTML = originalText;
            button.disabled = false;
            updateStats();
        }, 1500);
    });
});

// Initialize the application
initializeOptions();